//	TorusGames2DPool.c
//
//	© 2021 by Jeff Weeks
//	See TermsOfUse.txt

#include "TorusGames-Common.h"
#include "GeometryGamesLocalization.h"
#include "GeometryGamesSound.h"
#include <math.h>
#include <stdio.h>


#define NUM_2D_BACKGROUND_TEXTURE_REPETITIONS_POOL	8

#define BALL_RADIUS			0.04
#define HOLE_RADIUS			0.065
#define CUE_DEAD_ZONE 		(2.0 * BALL_RADIUS)
#define STICK_FACTOR		1.50	//	how long to draw the stick (for rigorously correct drawing,
									//		0.5*STICK_FACTOR - CUE_DEAD_ZONE shouldn't exceed 1.0)
#define GRAB_CUE_EPSILON	0.2		//	be generous with this, so most of the shaft is a valid grab target
#define SHOT_STRENGTH		2.0		//	(units/sec)/(cue length)
#define BALL_DECELERATION	0.05	//	units/sec²
#define FINAL_SPEED			0.1		//	final object ball speed when computer takes a shot
									//		0.001 works great, but a larger value makes the shot seem more human.
#define POCKET_FORCE		1.0		//	units/sec²  Within the pocket, balls accelerate towards the center with this acceleration.
#define POCKET_RESISTANCE	1.0		//	(units/sec²)/(units/sec) = 1/sec
#define ENDPOINT_EPSILON	0.0001
#define STICK_PLACE_TIME	1.0		//	time interval to place cue stick for computer's shot	
#define STICK_PAUSE_TIME	1.0		//	time interval to pause cue stick before computer's shot	
#define TIME_EPSILON		0.00001	//	will depend on how accurately we can compute collision times
#define VERY_LONG_TIME		1.0e8	//	impossibly long time interval
#define STRETCH_FACTOR		1.001
#define CUE_HOT_SPOT_SIZE	0.1

//	In the touch-based interface, when continuing a previously started cue stick movement,
//	how close (in board units) must the new touch be to the previous cue stick grab point 
//	to count as a hit?
#define POOL_AIM_CONTINUATION_HIT_RADIUS	0.1

//	In the touch-based interface, the user taps to confirm the cue stick position
//	and initiate the shot.  How close must the touch point stay to the cue stick's
//	hot spot in order to count as a tap?  POOL_TAP_TOLERANCE is currently set to zero,
//	to let the user make arbitrarily small adjustments without prematurely triggering the shot.
#define POOL_TAP_TOLERANCE					0.0

//	Provide a cue ball (0), three solid balls (1,2,3),
//	a black ball (4) and three striped balls (5,6,7).
#if NUM_POOL_BALLS != 8
	This pool implementation assumes NUM_POOL_BALLS == 8.
#endif


//	BallPair holds position and velocity data
//	for a pair of potentially colliding balls.
typedef struct
{
	double	xa,	//	position of Ball A
			ya,
			xb,	//	position of Ball B
			yb,
			ua,	//	velocity of Ball A
			va,
			ub,	//	velocity of Ball B
			vb;
	bool	fa,	//	Is Ball A flipped?
			fb;	//	Is Ball B flipped?
} BallPair;

//	BallPath describes a potential path that might prove useful
//	when the computer takes its shot (in human vs. computer mode).
typedef struct
{
	double	itsStartH,
			itsStartV,
			itsDirectionH,
			itsDirectionV,
			itsLength,
			itsNormalH,
			itsNormalV,
			itsStartSpeed;
} BallPath;

//	Complex multiplication is convenient for adding
//	a Gaussian error to the computer's shots.
typedef struct
{
	double	re,
			im;
} Complex;


//	Public function

//	Public functions with private names
static void		PoolReset(ModelData *md);
static void		PoolHumanVsComputerChanged(ModelData *md);
static bool		PoolDragBegin(ModelData *md, bool aRightClick);
static void		PoolDragObject(ModelData *md, double aHandLocalDeltaH, double aHandLocalDeltaV);
static void		PoolDragEnd(ModelData *md, double aDragDuration, bool aTouchSequenceWasCancelled);
static void		PoolSimulationUpdate(ModelData *md);
static void		PoolRefreshMessage(ModelData *md);

//	Private functions
static void		CueVector(ModelData *md, double *aCueVectorH, double *aCueVectorV, double *aCueVectorLength);
static void		SetCueHotSpotForVector(ModelData *md, double aCueVectorH, double aCueVectorV);
static void		Shoot1(ModelData *md);
static void		Shoot2(ModelData *md, double aTimeInterval);
static void		Shoot3(ModelData *md);
static bool		BallsAreRolling(ModelData *md);
static void		SinkBalls(ModelData *md);
static void		SinkOneBall(ModelData *md, unsigned int aBall);
static void		SimulateBallMotion(ModelData *md, double aTimeInterval);
static bool		GetFirstCollision(ModelData *md, double aTimeInterval, double *aCollisionTime, unsigned int *aBallA, unsigned int *aBallB);
static double	GetCollisionTime(ModelData *md, double aTimeInterval, unsigned int aBallA, unsigned int aBallB);
static void		SetBallPair(ModelData *md, BallPair *p, unsigned int aBallA, unsigned int aBallB);
static void		RollAllBalls(ModelData *md, double aTimeInterval);
static void		TransferMomentum(ModelData *md, unsigned int aBallA, unsigned int aBallB);
static void		PullIntoPocket(ModelData *md, double aTimeInterval);
static void		SlowTheBalls(ModelData *md, double aTimeInterval);
static void		RestoreBall(ModelData *md, unsigned int aBall);
static bool		BallPositionIsLegal(ModelData *md, unsigned int aBall);
static void		ComputerChoosesShot(ModelData *md);
static bool		ObjectBallIsValid(ModelData *md, unsigned int aBall);
static void		SetPath(BallPath *aPath, double h0, double v0, double h1, double v1, double s1);
static bool		PathIsClear(ModelData *md, BallPath *aPath, unsigned int aBall, bool aPocketTest);
static bool		PathAvoidsObstacle(BallPath *aPath, double anObstacleH, double anObstacleV, double aRadius);
static void		AddGaussianError(ModelData *md);
static double	GetRandomGaussian(void);
static Complex	ComplexMultiply(Complex z, Complex w);


void Pool2DSetUp(ModelData *md)
{
	//	Initialize function pointers.
	md->itsGameShutDown					= NULL;
	md->itsGameReset					= &PoolReset;
	md->itsGameHumanVsComputerChanged	= &PoolHumanVsComputerChanged;
	md->itsGame2DHandMoved				= NULL;
	md->itsGame2DDragBegin				= &PoolDragBegin;
	md->itsGame2DDragObject				= &PoolDragObject;
	md->itsGame2DDragEnd				= &PoolDragEnd;
	md->itsGame3DDragBegin				= NULL;
	md->itsGame3DDragObject				= NULL;
	md->itsGame3DDragEnd				= NULL;
	md->itsGame3DGridSize				= NULL;
	md->itsGameCharacterInput			= NULL;
	md->itsGameSimulationUpdate			= &PoolSimulationUpdate;
	md->itsGameRefreshMessage			= &PoolRefreshMessage;

	//	Check the compile time parameters.
	GEOMETRY_GAMES_ASSERT(
		0.5*STICK_FACTOR - CUE_DEAD_ZONE <= 1.0,
		"Bad Pool Parameters.  The Torus Games framework "
		"draws objects correctly iff they extend no further "
		"than 0.5 units beyond the boundary of the fundamental domain "
		"in which they are defined.");
	
	//	Initialize the game.
	PoolReset(md);
}


static void PoolReset(ModelData *md)
{
	unsigned int	i;

	//	Note:  With the balls packed tightly together,
	//	the program can hang on the break (very rarely, but it happens).
	//	To avoid this, introduce a STRETCH_FACTOR to space the balls
	//	very slightly apart.  Confession:  I haven't rigorously tested
	//	this fix, because the bug is not readily reproducible.

	double	theIntialBallPlacement[NUM_POOL_BALLS][2] =
	{
		//	cue ball
		{	0.0,										 0.25											},
		
		//	solid balls
		{	0.0  -      BALL_RADIUS * STRETCH_FACTOR,	-0.25  -  ROOT3 * BALL_RADIUS * STRETCH_FACTOR	},
		{	0.0  +  2 * BALL_RADIUS * STRETCH_FACTOR,	-0.25											},
		{	0.0  -      BALL_RADIUS * STRETCH_FACTOR,	-0.25  +  ROOT3 * BALL_RADIUS * STRETCH_FACTOR	},

		//	4-ball
		{	0.0,										-0.25											},
		
		//	striped balls
		{	0.0  +      BALL_RADIUS * STRETCH_FACTOR,	-0.25  +  ROOT3 * BALL_RADIUS * STRETCH_FACTOR	},
		{	0.0  -  2 * BALL_RADIUS * STRETCH_FACTOR,	-0.25											},
		{	0.0  +      BALL_RADIUS * STRETCH_FACTOR,	-0.25  -  ROOT3 * BALL_RADIUS * STRETCH_FACTOR	}
	};

	
	//	All balls in play.
	for (i = 0; i < NUM_POOL_BALLS; i++)
		md->itsGameOf.Pool2D.itsBallInPlay[i] = true;

	//	Place all balls.
	for (i = 0; i < NUM_POOL_BALLS; i++)
	{
		md->itsGameOf.Pool2D.itsBallPlacement[i].itsH		= theIntialBallPlacement[i][0];
		md->itsGameOf.Pool2D.itsBallPlacement[i].itsV		= theIntialBallPlacement[i][1];
		md->itsGameOf.Pool2D.itsBallPlacement[i].itsFlip	= false;
		md->itsGameOf.Pool2D.itsBallPlacement[i].itsAngle	= 0.0;
		md->itsGameOf.Pool2D.itsBallPlacement[i].itsSizeH	= 2.0 * BALL_RADIUS;
		md->itsGameOf.Pool2D.itsBallPlacement[i].itsSizeV	= 2.0 * BALL_RADIUS;
	}

#ifdef MAKE_GAME_CHOICE_ICONS

	//	With these ball placements, it works well to export
	//	the full image at 176×176 and then cut out a 64×64 square
	//	containing balls #1, #3 and #6.

	md->itsGameOf.Pool2D.itsBallPlacement[0].itsH = +0.25;
	md->itsGameOf.Pool2D.itsBallPlacement[0].itsV = +0.00;

	md->itsGameOf.Pool2D.itsBallPlacement[1].itsH = -0.25 + 0.10;
	md->itsGameOf.Pool2D.itsBallPlacement[1].itsV = +0.25 - 0.10;

	md->itsGameOf.Pool2D.itsBallPlacement[3].itsH = -0.25 + 0.05;
	md->itsGameOf.Pool2D.itsBallPlacement[3].itsV = +0.25 + 0.10;

	md->itsGameOf.Pool2D.itsBallPlacement[6].itsH = -0.25 - 0.10;
	md->itsGameOf.Pool2D.itsBallPlacement[6].itsV = +0.25 + 0.05;

#endif
	
	//	Set all ball velocities to zero.
	for (i = 0; i < NUM_POOL_BALLS; i++)
	{
		md->itsGameOf.Pool2D.itsBallVelocityH[i] = 0.0;
		md->itsGameOf.Pool2D.itsBallVelocityV[i] = 0.0;
	}
	
	//	Initialize the cue stick.
	md->itsGameOf.Pool2D.itsCueStatus						= PoolCueVisible;
	md->itsGameOf.Pool2D.itsCueBecameActiveWithCurrentDrag	= false;
	SetCueHotSpotForVector(md, 0.125, 0.250);	//	relative to cue ball position
	
	//	PlayerA goes first.
	md->itsGameOf.Pool2D.itsWhoseTurn = PoolPlayerA;
	
	//	We don't know yet who will shoot for solids
	//	and who will shoot for stripes.
	md->itsGameOf.Pool2D.itsPlayerAGoal = PoolUndecided;
	md->itsGameOf.Pool2D.itsPlayerBGoal = PoolUndecided;

	//	Nobody has yet won or lost.
	md->itsGameOf.Pool2D.itsWinFlag  = false;
	md->itsGameOf.Pool2D.itsLoseFlag = false;

	//	When playing human vs. computer,
	//	let the computer play for PoolPlayerB by default.
	md->itsGameOf.Pool2D.itsComputerPlayer = PoolPlayerB;

#ifdef GAME_CONTENT_FOR_SCREENSHOT

	//	Place the balls in a visually appealing arrangement
	//	to create screenshots for the App Store.

	md->itsGameOf.Pool2D.itsPlayerAGoal = PoolSolids;
	md->itsGameOf.Pool2D.itsPlayerBGoal = PoolStripes;

	md->itsGameOf.Pool2D.itsBallPlacement[0].itsH = -0.24;
	md->itsGameOf.Pool2D.itsBallPlacement[0].itsV = +0.23;	//	+0.23 for App Store image
//	md->itsGameOf.Pool2D.itsBallPlacement[0].itsV = -0.23;	//	-0.23 for How to Play image

	md->itsGameOf.Pool2D.itsBallPlacement[1].itsH =  0.01;
	md->itsGameOf.Pool2D.itsBallPlacement[1].itsV =  0.48;

	md->itsGameOf.Pool2D.itsBallPlacement[2].itsH =  0.49;
	md->itsGameOf.Pool2D.itsBallPlacement[2].itsV =  0.12;

	md->itsGameOf.Pool2D.itsBallPlacement[3].itsH =  0.20;
	md->itsGameOf.Pool2D.itsBallPlacement[3].itsV = -0.08;

	md->itsGameOf.Pool2D.itsBallPlacement[4].itsH =  0.09;
	md->itsGameOf.Pool2D.itsBallPlacement[4].itsV =  0.30;

	md->itsGameOf.Pool2D.itsBallInPlay[5] = false;

	md->itsGameOf.Pool2D.itsBallPlacement[6].itsH = -0.34;
	md->itsGameOf.Pool2D.itsBallPlacement[6].itsV =  0.44;

	md->itsGameOf.Pool2D.itsBallPlacement[7].itsH = -0.11;
	md->itsGameOf.Pool2D.itsBallPlacement[7].itsV = -0.38;

	md->itsGameOf.Pool2D.itsWhoseTurn = PoolPlayerB;

#endif

	//	Abort any pending simulation.
	SimulationEnd(md);
	
	//	Display the initial status message.
	PoolRefreshMessage(md);
	
	//	Go.
	md->itsGameIsOver = false;
}


static void PoolHumanVsComputerChanged(ModelData *md)
{
	//	If the user changed from human-vs-human to human-vs-computer...
	if (md->itsHumanVsComputer)
	{
		//	...then let computer play as the currently active player's opponents.
		md->itsGameOf.Pool2D.itsComputerPlayer = ! md->itsGameOf.Pool2D.itsWhoseTurn;
	}
}


static void PoolRefreshMessage(ModelData *md)
{
	const Char16	*theSubject,
					*thePredicate;
	Char16			theKey[64];

	static ColorP3Linear	thePoolMessageColor = {0.0, 0.125, 0.0, 1.0};

	//	Determine the active subject.
	if (md->itsHumanVsComputer)
	{
		if (md->itsGameOf.Pool2D.itsWhoseTurn == md->itsGameOf.Pool2D.itsComputerPlayer)
			theSubject = u"Computer";
		else
			theSubject = u"Human";
	}
	else
	{
		if (md->itsGameOf.Pool2D.itsWhoseTurn == PoolPlayerA)
			theSubject = u"PlayerA";
		else
			theSubject = u"PlayerB";
	}
		
	//	Determine the appropriate predicate.
	if (md->itsGameOf.Pool2D.itsWinFlag)
	{
		thePredicate = u"HasWon";
	}
	else
	if (md->itsGameOf.Pool2D.itsLoseFlag)
	{
		thePredicate = u"HasLost";
	}
	else
	{
		switch (md->itsGameOf.Pool2D.itsWhoseTurn == PoolPlayerA ?
					md->itsGameOf.Pool2D.itsPlayerAGoal :
					md->itsGameOf.Pool2D.itsPlayerBGoal)
		{
			case PoolUndecided:	thePredicate = u"Shoots";				break;
			case PoolSolids:	thePredicate = u"ShootsForSolids";		break;
			case PoolStripes:	thePredicate = u"ShootsForStripes";		break;
			case PoolBlack:		thePredicate = u"ShootsForFourBall";	break;
			default:			thePredicate = u"";						break;	//	unused
		}
	}

	//	Assemble theKey.
	Strcpy16(theKey, BUFFER_LENGTH(theKey), theSubject);
	Strcat16(theKey, BUFFER_LENGTH(theKey), thePredicate);

	//	Fetch the localized pool message and display it.
	SetTorusGamesStatusMessage(GetLocalizedText(theKey), thePoolMessageColor, Game2DPool);
}


static bool PoolDragBegin(
	ModelData	*md,
	bool		aRightClick)
{
	UNUSED_PARAMETER(aRightClick);
	
	if (md->itsGameIsOver)
		return false;
	
	switch (md->itsGameOf.Pool2D.itsCueStatus)
	{
		case PoolCueNone:

			return false;
		
		case PoolCueVisible:

			if (Shortest2DDistance(	md->its2DHandPlacement.itsH,
									md->its2DHandPlacement.itsV,
									md->itsGameOf.Pool2D.itsCueHotSpot.itsH,
									md->itsGameOf.Pool2D.itsCueHotSpot.itsV,
									md->itsTopology,
									NULL)
				< GRAB_CUE_EPSILON)
			{
				md->itsGameOf.Pool2D.itsCueStatus						= PoolCueActive;
				md->itsGameOf.Pool2D.itsCueBecameActiveWithCurrentDrag	= true;

#ifdef TORUS_GAMES_2D_TOUCH_INTERFACE
				//	The touch-based interface doesn't move itsCueHotSpot until it knows
				//	the drag is not a tap, so as not to disturb the user's carefully 
				//	aligned shot when s/he taps to execute it.
				md->itsGameOf.Pool2D.itsDragStartH	= md->its2DHandPlacement.itsH;
				md->itsGameOf.Pool2D.itsDragStartV	= md->its2DHandPlacement.itsV;
				md->itsGameOf.Pool2D.itsDragWasTap	= false;	//	A cue stick movement's first drag never counts as a terminating tap.
#else	//	mouse-based interface
				//	Let itsCueHotSpot track its2DHandPlacement immediately.
				md->itsGameOf.Pool2D.itsCueHotSpot	= md->its2DHandPlacement;
#endif

				return true;
			}
			else
			{
				//	Let the cue stick remain visible but inactive.
				return false;
			}
		
		case PoolCueActive:

			//	The touch-based and mouse-based versions
			//	of torus pool handle drags differently.

#ifdef TORUS_GAMES_2D_TOUCH_INTERFACE

			//	Let PoolDragEnd() know that the cue stick was active
			//	even before this drag began.
			md->itsGameOf.Pool2D.itsCueBecameActiveWithCurrentDrag = false;

			//	If the user has touched the cue's hot spot...
			if (Shortest2DDistance(
						md->its2DHandPlacement.itsH,
						md->its2DHandPlacement.itsV,
						md->itsGameOf.Pool2D.itsCueHotSpot.itsH,
						md->itsGameOf.Pool2D.itsCueHotSpot.itsV,
						md->itsTopology,
						NULL)
					< POOL_AIM_CONTINUATION_HIT_RADIUS)
			{
				//	...then let the pending cue stick movement continue.

				//	That is, start a new drag which will be part of the same pending cue stick movement.
				//	Note that we do not force itsCueHotSpot to move to its2DHandPlacement:
				//	a simple tap, even if slightly off-center, will serve to confirm 
				//	and initiate the intended shot without disturbing its direction.

				md->itsGameOf.Pool2D.itsDragStartH	= md->its2DHandPlacement.itsH;
				md->itsGameOf.Pool2D.itsDragStartV	= md->its2DHandPlacement.itsV;
				md->itsGameOf.Pool2D.itsDragWasTap	= true;

				return true;
			}
			else	//	But if the user touches elsewhere,
			{
				//	treat the touch as a scroll.
				//	The pre-existing cue stick movement remains valid but suspended,
				//	awaiting future touches for its final completion.
				return false;
			}

#else	//	mouse-based interface

			//	Moves are never left in suspension.
			return false;	//	should never occur

#endif
		
	}
}


static void PoolDragObject(
	ModelData	*md,
	double		aHandLocalDeltaH,
	double		aHandLocalDeltaV)
{
	//	Move the hand.
	md->its2DHandPlacement.itsH += (md->its2DHandPlacement.itsFlip ? -aHandLocalDeltaH : aHandLocalDeltaH);
	md->its2DHandPlacement.itsV += aHandLocalDeltaV;
	Normalize2DPlacement(&md->its2DHandPlacement, md->itsTopology);

#ifdef TORUS_GAMES_2D_TOUCH_INTERFACE

	//	If the cue stick's hot spot has travelled a non-trivial distance
	//	from the start of the present drag, then the drag is not a tap.
	//	A cue stick motion may consist of several separate drags, terminated by a tap.
	if (md->itsGameOf.Pool2D.itsDragWasTap)
	{
		if (Shortest2DDistance(
					md->itsGameOf.Pool2D.itsDragStartH,
					md->itsGameOf.Pool2D.itsDragStartV,
					md->its2DHandPlacement.itsH,
					md->its2DHandPlacement.itsV,
					md->itsTopology,
					NULL)
				> POOL_TAP_TOLERANCE)
		{
			md->itsGameOf.Pool2D.itsDragWasTap = false;
		}
	}

	//	The touch-based interface doesn't move itsCueHotSpot until it knows
	//	the drag is not a tap, so as not to disturb the user's carefully 
	//	aligned shot when s/he taps to execute it.
	if ( ! md->itsGameOf.Pool2D.itsDragWasTap )
		md->itsGameOf.Pool2D.itsCueHotSpot = md->its2DHandPlacement;

#else	//	mouse-based interface

	//	Let the cue stick's hot spot track the hand at all times.
	md->itsGameOf.Pool2D.itsCueHotSpot = md->its2DHandPlacement;

#endif
}


static void PoolDragEnd(
	ModelData	*md,
	double		aDragDuration,	//	in seconds
	bool		aTouchSequenceWasCancelled)
{
	UNUSED_PARAMETER(aDragDuration);

	//	If a touch sequence got cancelled (perhaps because a gesture was recognized)...
	if (aTouchSequenceWasCancelled)
	{
		//	If this touch sequence activated the cue stick
		//	(most likely unintentionally)...
		if (md->itsGameOf.Pool2D.itsCueBecameActiveWithCurrentDrag)
		{
			//	...then deactivate it (leaving it visible of course).
			md->itsGameOf.Pool2D.itsCueStatus						= PoolCueVisible;
			md->itsGameOf.Pool2D.itsCueBecameActiveWithCurrentDrag	= false;
		}
		else
		{
			//	The cue stick was active even before this touch sequence began,
			//	so let it remain active, awaiting subsequent drags.
		}
		
		//	Whether or not we just deactivated the cue stick, we're done handling this drag.
		return;
	}

#ifdef TORUS_GAMES_2D_TOUCH_INTERFACE
	if (md->itsGameOf.Pool2D.itsDragWasTap)
#else
	if (true)	//	Each move consists of a single drag.
#endif
	{
		//	Just be tidy, clear itsCueBecameActiveWithCurrentDrag.
		//	Shoot1() will decide what to do with itsCueStatus.
		md->itsGameOf.Pool2D.itsCueBecameActiveWithCurrentDrag = false;

		//	Animate the human's shot iff the pool cue's hot spot lies outside the dead zone.
		Shoot1(md);
	}
	else
	{
		//	The drag was not a simple tap.  Even though the drag is over,
		//	let the cue stick movement remain pending, awaiting subsequent drags.
	}
}


static void CueVector(
	ModelData	*md,
	double		*aCueVectorH,
	double		*aCueVectorV,
	double		*aCueVectorLength)
{
	double	theBallH,
			theBallV,
			theHotSpotH,
			theHotSpotV,
			theFactor;

	theBallH	= md->itsGameOf.Pool2D.itsBallPlacement[0].itsH;
	theBallV	= md->itsGameOf.Pool2D.itsBallPlacement[0].itsV;
	
	theHotSpotH = md->itsGameOf.Pool2D.itsCueHotSpot.itsH;
	theHotSpotV = md->itsGameOf.Pool2D.itsCueHotSpot.itsV;
	
	if (theHotSpotV - theBallV > +0.5)
	{
		theHotSpotV -= 1.0;
		if (md->itsTopology == Topology2DKlein)
			theHotSpotH = - theHotSpotH;
	}
	if (theHotSpotV - theBallV < -0.5)
	{
		theHotSpotV += 1.0;
		if (md->itsTopology == Topology2DKlein)
			theHotSpotH = - theHotSpotH;
	}
	
	if (theHotSpotH - theBallH > +0.5)
		theHotSpotH -= 1.0;
	if (theHotSpotH - theBallH < -0.5)
		theHotSpotH += 1.0;

	*aCueVectorH = theBallH - theHotSpotH;
	*aCueVectorV = theBallV - theHotSpotV;
	
	*aCueVectorLength = sqrt(	(*aCueVectorH) * (*aCueVectorH)
							  + (*aCueVectorV) * (*aCueVectorV) );

	if (*aCueVectorLength > CUE_DEAD_ZONE)
	{
		theFactor = (*aCueVectorLength - CUE_DEAD_ZONE) / (*aCueVectorLength);
		
		*aCueVectorH		*= theFactor;
		*aCueVectorV		*= theFactor;
		*aCueVectorLength	*= theFactor;
	}
	else
	{
		*aCueVectorH		= 0.0;
		*aCueVectorV		= 0.0;
		*aCueVectorLength	= 0.0;
	}
}

static void SetCueHotSpotForVector(
	ModelData	*md,
	double		aCueVectorH,
	double		aCueVectorV)
{
	double		theCueVectorLength;

	md->itsGameOf.Pool2D.itsCueHotSpot.itsH		= md->itsGameOf.Pool2D.itsBallPlacement[0].itsH;
	md->itsGameOf.Pool2D.itsCueHotSpot.itsV		= md->itsGameOf.Pool2D.itsBallPlacement[0].itsV;
	md->itsGameOf.Pool2D.itsCueHotSpot.itsFlip	= false;	//	ignored
	md->itsGameOf.Pool2D.itsCueHotSpot.itsAngle	= 0.0;		//	ignored
	md->itsGameOf.Pool2D.itsCueHotSpot.itsSizeH	= 1.0;		//	ignored
	md->itsGameOf.Pool2D.itsCueHotSpot.itsSizeV	= 1.0;		//	ignored

	theCueVectorLength = sqrt(aCueVectorH * aCueVectorH
							+ aCueVectorV * aCueVectorV);
	
	if (theCueVectorLength > 0.0)
	{
		aCueVectorH *= (theCueVectorLength + CUE_DEAD_ZONE) / theCueVectorLength;
		aCueVectorV *= (theCueVectorLength + CUE_DEAD_ZONE) / theCueVectorLength;
		
		md->itsGameOf.Pool2D.itsCueHotSpot.itsH -= aCueVectorH;
		md->itsGameOf.Pool2D.itsCueHotSpot.itsV -= aCueVectorV;
		
		Normalize2DPlacement(&md->itsGameOf.Pool2D.itsCueHotSpot, md->itsTopology);
	}
}


static void Shoot1(ModelData *md)
{
	double	theCueVectorH;
	double	theCueVectorV;
	double	theCueVectorLength;

	//	CueVector() returns a cue vector with the dead zone already subtracted off,
	//	so the shot will be valid iff theCueVectorLength > 0.
	CueVector(md, &theCueVectorH, &theCueVectorV, &theCueVectorLength);
	if (theCueVectorLength == 0.0)
	{
		//	Ignore null shots.
		md->itsGameOf.Pool2D.itsCueStatus = PoolCueVisible;
		SetCueHotSpotForVector(md, 0.125, 0.250);	//	relative to cue ball position
		return;
	}

	//	The current player (human or computer) has chosen a shot.
	//	Set the balls rolling!

	//	Play the cue-stick-strikes-cue-ball sound.
	EnqueueSoundRequest(u"PoolCue.wav");
	
	//	Hide the cue stick while the balls are rolling.
	md->itsGameOf.Pool2D.itsCueStatus = PoolCueNone;
	
	//	Keep track of what balls get sunk.
	//	Assume a passive scratch until a valid ball drops.
	md->itsGameOf.Pool2D.itsPassiveScratchFlag	= true;
	md->itsGameOf.Pool2D.itsActiveScratchFlag	= false;
	
	//	CueVector

	//	Start the cue ball rolling.
	//	Store the velocity relative to the ball's local coordinates
	//	(which may be flipped in a Klein bottle).
	md->itsGameOf.Pool2D.itsBallVelocityH[0] = SHOT_STRENGTH * theCueVectorH;
	md->itsGameOf.Pool2D.itsBallVelocityV[0] = SHOT_STRENGTH * theCueVectorV;
	if (md->itsGameOf.Pool2D.itsBallPlacement[0].itsFlip)
		md->itsGameOf.Pool2D.itsBallVelocityH[0] = - md->itsGameOf.Pool2D.itsBallVelocityH[0];
	
	//	Animate the balls (by calling Shoot2()) until all have stopped rolling,
	//	then finalize the shot (by calling Shoot3()).
	SimulationBegin(md, Simulation2DPoolRollBalls);
}


static void Shoot2(
	ModelData	*md,
	double		aTimeInterval)	//	in seconds
{
	//	The balls are in motion.
	//	Advance the simulation by one frame.

	//	Is any ball completely over the pocket?
	//	(Note that we sink a ball just after letting the UI draw it,
	//	rather than just before, so that the user sees the ball
	//	fall all the way into the hole before it disappears.)
	SinkBalls(md);

	//	Simulate the balls' physics through aTimeInterval.
	SimulateBallMotion(md, aTimeInterval);
	
	//	The pocket attracts balls within it.
	PullIntoPocket(md, aTimeInterval);
	
	//	Rolling resistance slows the balls.
	SlowTheBalls(md, aTimeInterval);
}


static void Shoot3(ModelData *md)
{
	//	The balls have come to rest.
	//	Evaluate the result.

	//	What balls dropped?
	if (md->itsGameOf.Pool2D.itsWinFlag)
	{
		md->itsGameIsOver = true;
		EnqueueSoundRequest(u"PoolWin.mid");
	}
	else
	if (md->itsGameOf.Pool2D.itsLoseFlag)
	{
		md->itsGameIsOver = true;
		EnqueueSoundRequest(u"PoolLoss.mid");
	}
	else
	{
		if (md->itsGameOf.Pool2D.itsPassiveScratchFlag || md->itsGameOf.Pool2D.itsActiveScratchFlag)
			md->itsGameOf.Pool2D.itsWhoseTurn = ! md->itsGameOf.Pool2D.itsWhoseTurn;
	}

	//	Update the status message.
	PoolRefreshMessage(md);
	
	//	Restore the cue ball if necessary.
	if ( ! md->itsGameOf.Pool2D.itsBallInPlay[0] )
		RestoreBall(md, 0);

	//	Show the cue stick.
	md->itsGameOf.Pool2D.itsCueStatus = PoolCueVisible;

	//	If it's the computer's turn to shoot, let it
	//	(regardless of whether it or the human took the preceding shot).
	if ( ! md->itsGameIsOver
	 && md->itsHumanVsComputer
	 && md->itsGameOf.Pool2D.itsWhoseTurn == md->itsGameOf.Pool2D.itsComputerPlayer)
	{
		//	Choose a shot and execute it.
		ComputerChoosesShot(md);
		AddGaussianError(md);
		SimulationBegin(md, Simulation2DPoolPlaceCueStick);
	}
	else
	{
		//	Set a default cue stick position in preparation for the human's turn.
		SetCueHotSpotForVector(md, 0.125, 0.250);	//	relative to cue ball position
	}
}


static bool BallsAreRolling(ModelData *md)
{
	unsigned int	i;

	for (i = 0; i < NUM_POOL_BALLS; i++)
	{
		if (md->itsGameOf.Pool2D.itsBallInPlay[i]
		 && (	md->itsGameOf.Pool2D.itsBallVelocityH[i] != 0.0
			 || md->itsGameOf.Pool2D.itsBallVelocityV[i] != 0.0))
		{
			return true;
		}
	}

	return false;
}


static void SinkBalls(ModelData *md)
{
	unsigned int	i,
					theNumBallsOverPocket;
	double			theDistanceFromHole[NUM_POOL_BALLS],
					theSinkRadius;
	
	//	On rare occasions two balls can jam against each other
	//	in the pocket, with neither ball wholly contained therein.
	//	In and of itself that's not a problem, but in practice
	//	the pocket's attractive force makes the two balls bounce
	//	against each other, and thus the simulation never
	//	terminates because the balls never stop moving.
	//	To avoid this problem, make a rule that if the centers
	//	of two balls are simultaneously in the pocket,
	//	both balls get sunk.  Yes, I know this is a real hack,
	//	but it seems to be the cleanest solution possible.

	//	We'll count how many balls have their center-of-mass over the pocket.
	theNumBallsOverPocket = 0;

	//	Precompute each ball's distance from the hole.
	for (i = 0; i < NUM_POOL_BALLS; i++)
	{
		if (md->itsGameOf.Pool2D.itsBallInPlay[i])
		{
			theDistanceFromHole[i] = sqrt(	md->itsGameOf.Pool2D.itsBallPlacement[i].itsH * md->itsGameOf.Pool2D.itsBallPlacement[i].itsH
										  + md->itsGameOf.Pool2D.itsBallPlacement[i].itsV * md->itsGameOf.Pool2D.itsBallPlacement[i].itsV);

			if (theDistanceFromHole[i] < HOLE_RADIUS)
				theNumBallsOverPocket++;
		}
		else
		{
			theDistanceFromHole[i] = 0.0;	//	unused
		}
	}
	
	//	If two balls sit over the pocket both get sunk immediately (cf. above).
	//	If only one ball sits over the pocket, sink it only if it's fallen
	//	all the way in.
	if (theNumBallsOverPocket >= 2)
		theSinkRadius = HOLE_RADIUS;
	else
		theSinkRadius = HOLE_RADIUS - BALL_RADIUS;
	
	//	Sink a ball if it's within theSinkRadius.
	for (i = 0; i < NUM_POOL_BALLS; i++)
	{
		if (md->itsGameOf.Pool2D.itsBallInPlay[i]
		 && theDistanceFromHole[i] < theSinkRadius)
		{
			SinkOneBall(md, i);
		}
	}
}


static void SinkOneBall(
	ModelData		*md,
	unsigned int	aBall)
{
	PoolGoal	theGoal;

	//	Play the sinking sound.
	EnqueueSoundRequest(u"PoolSink.wav");

	//	Which balls is the current player shooting for?
	theGoal = (md->itsGameOf.Pool2D.itsWhoseTurn == PoolPlayerA ?
				md->itsGameOf.Pool2D.itsPlayerAGoal : md->itsGameOf.Pool2D.itsPlayerBGoal);

	//	Before sinking this ball,
	//	figure out what it will mean for the game.
	if (theGoal == PoolUndecided)
	{
		switch (aBall)
		{
			case 0:		//	cue ball
				md->itsGameOf.Pool2D.itsActiveScratchFlag = true;
				break;
			
			case 1:		//	solid ball
			case 2:
			case 3:
				if (md->itsGameOf.Pool2D.itsBallInPlay[0] && md->itsGameOf.Pool2D.itsBallInPlay[4])
				{
					if (md->itsGameOf.Pool2D.itsWhoseTurn == PoolPlayerA)
					{
						md->itsGameOf.Pool2D.itsPlayerAGoal = PoolSolids;
						md->itsGameOf.Pool2D.itsPlayerBGoal = PoolStripes;
					}
					else
					{
						md->itsGameOf.Pool2D.itsPlayerAGoal = PoolStripes;
						md->itsGameOf.Pool2D.itsPlayerBGoal = PoolSolids;
					}
					md->itsGameOf.Pool2D.itsPassiveScratchFlag = false;
				}
				break;
			
			case 4:		//	4-ball
				md->itsGameOf.Pool2D.itsLoseFlag = true;
				break;

			case 5:		//	striped ball
			case 6:
			case 7:
				if (md->itsGameOf.Pool2D.itsBallInPlay[0] && md->itsGameOf.Pool2D.itsBallInPlay[4])
				{
					if (md->itsGameOf.Pool2D.itsWhoseTurn == PoolPlayerA)
					{
						md->itsGameOf.Pool2D.itsPlayerAGoal = PoolStripes;
						md->itsGameOf.Pool2D.itsPlayerBGoal = PoolSolids;
					}
					else
					{
						md->itsGameOf.Pool2D.itsPlayerAGoal = PoolSolids;
						md->itsGameOf.Pool2D.itsPlayerBGoal = PoolStripes;
					}
					md->itsGameOf.Pool2D.itsPassiveScratchFlag = false;
				}
				break;
		}
	}
	else	//	itsGameOf.Pool2D.itsPlayerGoals (solids, stripes or 4-ball) have already been decided
	{
		if ( ! md->itsGameOf.Pool2D.itsBallInPlay[4] )		//	4-ball already sunk
		{
			//	If the player has already sunk the 4-ball,
			//	but then some other ball follows it in,
			//	we must cancel the pending win
			//	and declare a loss instead.
			md->itsGameOf.Pool2D.itsWinFlag  = false;
			md->itsGameOf.Pool2D.itsLoseFlag = true;
		}
		else if ( ! md->itsGameOf.Pool2D.itsBallInPlay[0] )	//	cue ball already sunk
		{
			if (aBall == 4)
				md->itsGameOf.Pool2D.itsLoseFlag	= true;
		}
		else	//	neither 4-ball nor cue ball previously sunk
		{
			switch (aBall)
			{
				case 0:		//	cue ball
					md->itsGameOf.Pool2D.itsActiveScratchFlag = true;
					break;
				
				case 1:		//	solid ball
				case 2:
				case 3:
					if (theGoal == PoolSolids)
						md->itsGameOf.Pool2D.itsPassiveScratchFlag = false;
					else
						md->itsGameOf.Pool2D.itsActiveScratchFlag  = true;
					break;
				
				case 4:		//	4-ball
					if (theGoal == PoolBlack)
						md->itsGameOf.Pool2D.itsWinFlag  = true;
					else
						md->itsGameOf.Pool2D.itsLoseFlag = true;
					break;

				case 5:		//	striped ball
				case 6:
				case 7:
					if (theGoal == PoolStripes)
						md->itsGameOf.Pool2D.itsPassiveScratchFlag = false;
					else
						md->itsGameOf.Pool2D.itsActiveScratchFlag  = true;
					break;
			}
		}
	}
	
	//	Sink the ball.
	md->itsGameOf.Pool2D.itsBallInPlay[aBall] = false;
	
	//	Has either player qualified to shoot for the black ball?
	if ( ! (md->itsGameOf.Pool2D.itsBallInPlay[1] ||
			md->itsGameOf.Pool2D.itsBallInPlay[2] ||
			md->itsGameOf.Pool2D.itsBallInPlay[3]	))
	{
		if (md->itsGameOf.Pool2D.itsPlayerAGoal == PoolSolids)
			md->itsGameOf.Pool2D.itsPlayerAGoal = PoolBlack;
		if (md->itsGameOf.Pool2D.itsPlayerBGoal == PoolSolids)
			md->itsGameOf.Pool2D.itsPlayerBGoal = PoolBlack;
	}
	if ( ! (md->itsGameOf.Pool2D.itsBallInPlay[5] ||
			md->itsGameOf.Pool2D.itsBallInPlay[6] ||
			md->itsGameOf.Pool2D.itsBallInPlay[7]	))
	{
		if (md->itsGameOf.Pool2D.itsPlayerAGoal == PoolStripes)
			md->itsGameOf.Pool2D.itsPlayerAGoal = PoolBlack;
		if (md->itsGameOf.Pool2D.itsPlayerBGoal == PoolStripes)
			md->itsGameOf.Pool2D.itsPlayerBGoal = PoolBlack;
	}
}


static void SimulateBallMotion(
	ModelData	*md,
	double		aTimeInterval)	//	in seconds
{
	double			theCollisionTime;
	unsigned int	theBallA	= 0,
					theBallB	= 0;

	while (GetFirstCollision(md, aTimeInterval, &theCollisionTime, &theBallA, &theBallB))
	{
		RollAllBalls(md, theCollisionTime);
		TransferMomentum(md, theBallA, theBallB);
		aTimeInterval -= theCollisionTime;
	}

	RollAllBalls(md, aTimeInterval);
}


static bool GetFirstCollision(
	ModelData		*md,
	double			aTimeInterval,		//	input, in seconds
	double			*aCollisionTime,	//	output (valid iff function returns true)
	unsigned int	*aBallA,			//	output (valid iff function returns true)
	unsigned int	*aBallB)			//	output (valid iff function returns true)
{
	//	In principle we want to test for collisions at times in the range
	//	[0, aTimeInterval].  In practice we extend the range to
	//	[-TIME_EPSILON, aTimeInterval + TIME_EPSILON] to make sure
	//	no collisions slip through the cracks.  We won't have to worry
	//	about seeing the same collision twice if we consider only collisions
	//	in which the balls are moving towards each other.

	double			theCollisionTime;
	unsigned int	theBallA,
					theBallB;
	
	//	Initialize *aCollisionTime to an impossibly large value.
	*aCollisionTime = VERY_LONG_TIME;
	
	//	Consider all pairs of balls.
	for (theBallA = 0; theBallA < NUM_POOL_BALLS; theBallA++)
	{
		if (md->itsGameOf.Pool2D.itsBallInPlay[theBallA])
		{
			for (theBallB = theBallA + 1; theBallB < NUM_POOL_BALLS; theBallB++)
			{
				if (md->itsGameOf.Pool2D.itsBallInPlay[theBallB])
				{
					theCollisionTime = GetCollisionTime(md, aTimeInterval, theBallA, theBallB);
					if (theCollisionTime < *aCollisionTime)
					{
						*aCollisionTime	= theCollisionTime;
						*aBallA			= theBallA;
						*aBallB			= theBallB;
					}
				}
			}
		}
	}
	
	//	Report the collision or lack thereof.
	if (*aCollisionTime < aTimeInterval + TIME_EPSILON)
	{
		return true;
	}
	else
	{
		*aCollisionTime	= 0;
		*aBallA			= 0;
		*aBallB			= 0;
		return false;
	}
}


static double GetCollisionTime(
	ModelData		*md,
	double			aTimeInterval,	//	in seconds
	unsigned int	aBallA,
	unsigned int	aBallB)
{
	//	If Ball A and Ball B will collide within the time interval
	//	[-TIME_EPSILON, aTimeInterval + TIME_EPSILON], return the time
	//	of the inbound collision.  Otherwise return a VERY_LONG_TIME.
	
	BallPair	p;
	double		a,
				b,
				c,
				theDiscriminant,
				t;
	
	//	Locate images of BallA and BallB as close to each other as possible.
	SetBallPair(md, &p, aBallA, aBallB);
	
	//	The two balls will overlap when the difference vector
	//
	//		[(xa,ya) + t*(ua,va)] - [(xb,yb) + t*(ub,vb)]
	//
	//	has length less than 2*BALL_RADIUS.  Expanding that expression gives
	//
	//		[(xa - xb) + t*(ua - ub), (ya - yb) + t*(va - vb)]
	//
	//	whose squared length is
	//
	//		 t² * [(ua - ub)² + (va - vb)²]
	//	  +  t  * [2*(xa - xb)*(ua - ub) + 2*(ya - yb)*(va - vb)]
	//	  +  1  * [(xa - xb)² + (ya - yb)²].
	//
	//	Set that last expression equal to (2*BALL_RADIUS)²
	//	and write down the coefficients of the quadratic equation
	//
	//		a t² + b t + c = 0.
	a = (p.ua - p.ub)*(p.ua - p.ub) + (p.va - p.vb)*(p.va - p.vb);
	b = 2*(p.xa - p.xb)*(p.ua - p.ub) + 2*(p.ya - p.yb)*(p.va - p.vb);
	c = (p.xa - p.xb)*(p.xa - p.xb) + (p.ya - p.yb)*(p.ya - p.yb) 
		- 4 * BALL_RADIUS * BALL_RADIUS;
	
	//	In the typical case, the discriminant will be negative and
	//	the balls will not collide.
	theDiscriminant = b*b - 4*a*c;
	if (theDiscriminant <= 0.0)
		return VERY_LONG_TIME;
	
	//	Ignore the special case that the balls are travelling
	//	parallel to each other.  This case will occur in the original
	//	set-up (with seven motionless balls tangent to each other)
	//	but will otherwise be rare.
	if (a == 0.0)	//	could test against an epsilon, but maybe it isn't necessary
		return VERY_LONG_TIME;
	
	//	At this point we know the discriminant is positive,
	//	so we'll have an "inbound" collision (which we want)
	//	followed by an "outbound" collision (which we ignore).
	//	We know a > 0, so we can take the - part of the +/-
	//	in the quadratic formula.
	t = (-b - sqrt(theDiscriminant)) / (2*a);
	
	//	Does the collision occur within the time interval
	//	[-TIME_EPSILON, aTimeInterval + TIME_EPSILON] ?
	if (t > -TIME_EPSILON  &&  t < aTimeInterval + TIME_EPSILON)
		return t;
	else
		return VERY_LONG_TIME;
}


static void SetBallPair(
	ModelData		*md,
	BallPair		*p,
	unsigned int	aBallA,
	unsigned int	aBallB)
{
	//	Where are Ball A and Ball B now?
	p->xa = md->itsGameOf.Pool2D.itsBallPlacement[aBallA].itsH;
	p->ya = md->itsGameOf.Pool2D.itsBallPlacement[aBallA].itsV;
	p->xb = md->itsGameOf.Pool2D.itsBallPlacement[aBallB].itsH;
	p->yb = md->itsGameOf.Pool2D.itsBallPlacement[aBallB].itsV;
	
	//	Note the balls' velocities in their own local ball coordinates...
	p->ua = md->itsGameOf.Pool2D.itsBallVelocityH[aBallA];
	p->va = md->itsGameOf.Pool2D.itsBallVelocityV[aBallA];
	p->ub = md->itsGameOf.Pool2D.itsBallVelocityH[aBallB];
	p->vb = md->itsGameOf.Pool2D.itsBallVelocityV[aBallB];
	
	//	...then convert to global fundamental domain coordinates.
	p->fa = md->itsGameOf.Pool2D.itsBallPlacement[aBallA].itsFlip;
	if (p->fa)
		p->ua = - p->ua;
	p->fb = md->itsGameOf.Pool2D.itsBallPlacement[aBallB].itsFlip;
	if (p->fb)
		p->ub = - p->ub;
	
	//	Leaving Ball A where it is, find the closest translate of Ball B.
	//	The two balls must be quite close in order to interact within aTimeInterval
	//	(so we can safely be loose with the definition of "nearest")
	//	but we do need to allow for the possibility that the two balls
	//	straddle a boundary of the fundamental domain.
	if (p->yb - p->ya > +0.5)
	{
		p->yb -= 1.0;
		if (md->itsTopology == Topology2DKlein)
		{
			p->xb = - p->xb;
			p->ub = - p->ub;
			p->fb = ! p->fb;
		}
	}
	if (p->yb - p->ya < -0.5)
	{
		p->yb += 1.0;
		if (md->itsTopology == Topology2DKlein)
		{
			p->xb = - p->xb;
			p->ub = - p->ub;
			p->fb = ! p->fb;
		}
	}
	if (p->xb - p->xa > +0.5)
		p->xb -= 1.0;
	if (p->xb - p->xa < -0.5)
		p->xb += 1.0;
}


static void RollAllBalls(
	ModelData	*md,
	double		aTimeInterval)	//	in seconds
{
	unsigned int	i;
	double			theVelocityH,
					theVelocityV;

	for (i = 0; i < NUM_POOL_BALLS; i++)
	{
		if (md->itsGameOf.Pool2D.itsBallInPlay[i])
		{
			//	Convert the ball's velocity from the ball's local coordinates
			//	to global fundamental domain coordinates.
			theVelocityH = md->itsGameOf.Pool2D.itsBallVelocityH[i];
			theVelocityV = md->itsGameOf.Pool2D.itsBallVelocityV[i];
			if (md->itsGameOf.Pool2D.itsBallPlacement[i].itsFlip)
				theVelocityH = - theVelocityH;
		
			//	Move the ball.
			md->itsGameOf.Pool2D.itsBallPlacement[i].itsH += theVelocityH * aTimeInterval;
			md->itsGameOf.Pool2D.itsBallPlacement[i].itsV += theVelocityV * aTimeInterval;
			
			//	Normalize the ball's position.
			Normalize2DPlacement(&md->itsGameOf.Pool2D.itsBallPlacement[i], md->itsTopology);
		}
	}
}


static void TransferMomentum(
	ModelData		*md,
	unsigned int	aBallA,
	unsigned int	aBallB)
{
	//	Assume Ball A and Ball B are just now colliding.
	//	Transfer momentum from one to the other in such a way that
	//
	//		1.	The momentum transfer is parallel to the vector running
	//			from the center of Ball A to the center of Ball B.
	//
	//		2.	Energy is conserved.
	
	BallPair	p;
	double		dx,
				dy,
				theLength,
				dp;

	//	Play the ball clicking sound.
	EnqueueSoundRequest(u"PoolClick.wav");

	//	Locate images of BallA and BallB as close to each other as possible.
	SetBallPair(md, &p, aBallA, aBallB);
	
	//	Compute a unit vector running from the center of Ball A
	//	to the center of Ball B.
	dx = p.xb - p.xa;
	dy = p.yb - p.ya;
	theLength = sqrt(dx*dx + dy*dy);
	dx /= theLength;
	dy /= theLength;
	
	//	We want to transfer a (possibly negative) quantity dp of momentum
	//	from Ball A to Ball B along the unit vector (dx,dy).
	//	That is, Ball B will gain momentum dp*(dx,dy)
	//	while Ball A looses that same amount.
	//	The total kinetic energy before the collision (assuming all balls have
	//	unit mass) is
	//
	//		(1/2)*[ua² + va² + ub² + vb²].
	//
	//	The total kinetic energy after the collision will be
	//
	//		(1/2)*[(ua - dp*dx)² + (va - dp*dy)² + (ub + dp*dx)² + (vb + dp*dy)²].
	//
	//	Expanding those two expressions and setting them equal gives
	//
	//		dp² + dp*[dx*(ub - ua) + dy*(vb - va)] = 0.
	//
	//	Ignoring the trivial solution dp == 0 gives the desired
	//
	//		dp = dx*(ua - ub) + dy*(va - vb).

	dp = dx*(p.ua - p.ub) + dy*(p.va - p.vb);
	
	p.ua -= dp*dx;
	p.va -= dp*dy;
	
	p.ub += dp*dx;
	p.vb += dp*dy;

	//	Copy the final velocities from the BallPair's global coordinates
	//	back to each ball's local coordinates.
	md->itsGameOf.Pool2D.itsBallVelocityH[aBallA] = (p.fa ? - p.ua : p.ua);
	md->itsGameOf.Pool2D.itsBallVelocityV[aBallA] = p.va;
	md->itsGameOf.Pool2D.itsBallVelocityH[aBallB] = (p.fb ? - p.ub : p.ub);
	md->itsGameOf.Pool2D.itsBallVelocityV[aBallB] = p.vb;
}


static void PullIntoPocket(
	ModelData	*md,
	double		aTimeInterval)	//	in seconds
{
	//	When a ball rolls over the pocket,
	//	bend its trajectory towards the pocket's center.

	unsigned int	i;
	double			theRadius,
					theFactor,
					theDirectionH,
					theDirectionV,
					theSpeedIncrement;

	for (i = 0; i < NUM_POOL_BALLS; i++)
	{
		if (md->itsGameOf.Pool2D.itsBallInPlay[i])
		{
			//	The pocket sits at coordinates (0,0), which simplifies the computations.

			//	How far is the ball from the hole (center-to-center) ?
			theRadius = sqrt(	md->itsGameOf.Pool2D.itsBallPlacement[i].itsH * md->itsGameOf.Pool2D.itsBallPlacement[i].itsH
							  + md->itsGameOf.Pool2D.itsBallPlacement[i].itsV * md->itsGameOf.Pool2D.itsBallPlacement[i].itsV);

			//	If the ball lies over the hole...
			if (theRadius < HOLE_RADIUS)
			{
				//	Let (theDirectionH, theDirectionV) be a unit vector
				//	pointing from the ball to the center of the pocket.
				if (theRadius > 0.01 * HOLE_RADIUS)
					theFactor = -1.0 / theRadius;
				else
					theFactor = 0.0;	//	special case to avoid division by zero or other very small numbers
				theDirectionH = theFactor * md->itsGameOf.Pool2D.itsBallPlacement[i].itsH;
				theDirectionV = theFactor * md->itsGameOf.Pool2D.itsBallPlacement[i].itsV;
				
				//	Convert the direction to the ball's local coordinate system.
				if (md->itsGameOf.Pool2D.itsBallPlacement[i].itsFlip)
					theDirectionH = - theDirectionH;
				
				//	Accelerate the ball towards the center of the pocket.
				theSpeedIncrement = POCKET_FORCE * aTimeInterval;
				md->itsGameOf.Pool2D.itsBallVelocityH[i] += theSpeedIncrement * theDirectionH;
				md->itsGameOf.Pool2D.itsBallVelocityV[i] += theSpeedIncrement * theDirectionV;
			}
		}
	}
}


static void SlowTheBalls(
	ModelData	*md,
	double		aTimeInterval)	//	in seconds
{
	unsigned int	i;
	double			theSpeed,
					theDecrement,
					theFactor;

	for (i = 0; i < NUM_POOL_BALLS; i++)
	{
		if (md->itsGameOf.Pool2D.itsBallInPlay[i])
		{
			theSpeed = sqrt(md->itsGameOf.Pool2D.itsBallVelocityH[i]*md->itsGameOf.Pool2D.itsBallVelocityH[i] + md->itsGameOf.Pool2D.itsBallVelocityV[i]*md->itsGameOf.Pool2D.itsBallVelocityV[i]);

			//	Is the ball over the pocket?
			if (md->itsGameOf.Pool2D.itsBallPlacement[i].itsH * md->itsGameOf.Pool2D.itsBallPlacement[i].itsH
			  + md->itsGameOf.Pool2D.itsBallPlacement[i].itsV * md->itsGameOf.Pool2D.itsBallPlacement[i].itsV
			  < HOLE_RADIUS * HOLE_RADIUS)
			{
				//	Ball is over pocket.
				//
				//	Apply a deceleration proportional to the ball's speed,
				//	so fast-moving balls will slow down a lot,
				//	while slow-moving balls remain largely unaffected.
				//	In particular, balls drifting towards the pocket's center
				//	will not grind to a halt.
				theDecrement = (POCKET_RESISTANCE * theSpeed) * aTimeInterval;
			}
			else
			{
				//	Ball is not over pocket.
				//
				//	Apply a constant deceleration.
				theDecrement = BALL_DECELERATION * aTimeInterval;
			}

			if (theSpeed <= theDecrement)
			{
				md->itsGameOf.Pool2D.itsBallVelocityH[i] = 0.0;
				md->itsGameOf.Pool2D.itsBallVelocityV[i] = 0.0;
			}
			else
			{
				theFactor = (theSpeed - theDecrement) / theSpeed;
				md->itsGameOf.Pool2D.itsBallVelocityH[i] *= theFactor;
				md->itsGameOf.Pool2D.itsBallVelocityV[i] *= theFactor;
			}
		}
	}
}


static void RestoreBall(
	ModelData		*md,
	unsigned int	aBall)
{
	//	Keep trying random locations until the ball
	//	overlaps neither the pocket nor another ball.
	do
	{
		md->itsGameOf.Pool2D.itsBallPlacement[aBall].itsH		= RandomFloat() - 0.5;
		md->itsGameOf.Pool2D.itsBallPlacement[aBall].itsV		= RandomFloat() - 0.5;
		md->itsGameOf.Pool2D.itsBallPlacement[aBall].itsFlip	= false;

	} while ( ! BallPositionIsLegal(md, aBall) );

	//	Set the ball's velocity to zero.
	md->itsGameOf.Pool2D.itsBallVelocityH[aBall] = 0.0;
	md->itsGameOf.Pool2D.itsBallVelocityV[aBall] = 0.0;
	
	//	Put the ball into play.
	md->itsGameOf.Pool2D.itsBallInPlay[aBall] = true;
}


static bool BallPositionIsLegal(
	ModelData		*md,
	unsigned int	aBall)
{
	unsigned int	i;
	BallPair		p;
	double			dx,
					dy;

	//	Don't overlap the pocket.
	if (md->itsGameOf.Pool2D.itsBallPlacement[aBall].itsH * md->itsGameOf.Pool2D.itsBallPlacement[aBall].itsH
	  + md->itsGameOf.Pool2D.itsBallPlacement[aBall].itsV * md->itsGameOf.Pool2D.itsBallPlacement[aBall].itsV
	  < (HOLE_RADIUS + BALL_RADIUS)*(HOLE_RADIUS + BALL_RADIUS))
	{
		return false;
	}

	//	Don't overlap another ball.
	for (i = 0; i < NUM_POOL_BALLS; i++)
	{
		if (i != aBall
		 && md->itsGameOf.Pool2D.itsBallInPlay[i])
		{
			SetBallPair(md, &p, aBall, i);
			
			dx = p.xb - p.xa;
			dy = p.yb - p.ya;

			if (dx*dx + dy*dy < (2.0 * BALL_RADIUS)*(2.0 * BALL_RADIUS))
				return false;
		}
	}
	
	//	The ball's position is OK.
	return true;
}


static void ComputerChoosesShot(ModelData *md)
{
	//	The computer selects its shot (when playing human vs. computer)
	//	and sets itsCueHotSpot accordingly.
	//	It doesn't actually make the shot -- the usual Shoot1() function 
	//	will take care of that.
	
	double			theMaxCueVectorLength,
					theMaxStartSpeed,
					theBestStartSpeed;
	unsigned int	theBall;
	signed int		thePocketH,
					thePocketV,
					dh,
					dv;
	double			theFirstLegEndH,
					theFirstLegEndV,
					theCueBallH,
					theCueBallV,
					theDotProduct,
					theFinalCueBallSpeed;
	BallPath		theFirstLeg,
					theSecondLeg;
	
	//	Evaluate the various possible shots, choosing the one that
	//	minimizes the cue ball's initial speed.  If no shot can be
	//	realized within the allowable MaxStartSpeed, we'll invoke
	//	a fallback plan below.
	theMaxCueVectorLength	= 0.5 - CUE_DEAD_ZONE;
	theMaxStartSpeed		= SHOT_STRENGTH * theMaxCueVectorLength;
	theBestStartSpeed		= theMaxStartSpeed;

	//	Consider each possible object ball.
	for (theBall = 0; theBall < NUM_POOL_BALLS; theBall++)
	{
		//	Is this object ball a valid target?
		if ( ! ObjectBallIsValid(md, theBall) )
			continue;
		
		//	Use the coordinate system of the tiling view ("universal cover").
		//	The object ball sits in the central copy of the fundamental domain,
		//	but we may want to make use of images of the cue ball and/or
		//	the pocket sitting in nearby copies of the fundamental domain.
		
		//	Consider 9 possible images of the pocket.
		for (thePocketH = -1; thePocketH <= +1; thePocketH++)
			for (thePocketV = -1; thePocketV <= +1; thePocketV++)
			{
				//	Let theSecondLeg be the path from the object ball
				//	to the given image of the pocket.
				SetPath(&theSecondLeg,
						md->itsGameOf.Pool2D.itsBallPlacement[theBall].itsH, md->itsGameOf.Pool2D.itsBallPlacement[theBall].itsV,
						thePocketH, thePocketV,
						FINAL_SPEED);

				//	If anything obstructs theSecondLeg,
				//	give up on it and move on to the next candidate.
				if ( ! PathIsClear(md, &theSecondLeg, theBall, false) )
					continue;

				//	The end of the first leg should be colinear with the second leg,
				//	but one ball diameter before the second leg's beginning.
				//	(Draw a picture and all will be clear!)
				
				theFirstLegEndH = theSecondLeg.itsStartH 
								- (2.0 * BALL_RADIUS) * theSecondLeg.itsDirectionH;
				theFirstLegEndV = theSecondLeg.itsStartV 
								- (2.0 * BALL_RADIUS) * theSecondLeg.itsDirectionV;
				
				//	Consider 9 possible images of the cue ball.
				for (dh = -1; dh <= +1; dh++)
					for (dv = -1; dv <= +1; dv++)
					{
						theCueBallH = md->itsGameOf.Pool2D.itsBallPlacement[0].itsH + dh;
						theCueBallV = md->itsGameOf.Pool2D.itsBallPlacement[0].itsV + dv;
						if (md->itsTopology == Topology2DKlein
						 && dv != 0)
							theCueBallH = - theCueBallH;

						//	Set the cue ball's path,
						//	but with a temporary final speed of zero.
						SetPath(&theFirstLeg,
								theCueBallH, theCueBallV,
								theFirstLegEndH, theFirstLegEndV,
								0.0);

						//	A shot is possible iff the path is clear...
						if ( ! PathIsClear(md, &theFirstLeg, 0, true) )
							continue;
						
						//	..and the dot product of the direction vectors
						//	for the two legs is positive.  (A negative dot product
						//	represents an impossible shot, because the cue ball
						//	would have to pass through the object ball on the first leg.)
						theDotProduct = theFirstLeg.itsDirectionH * theSecondLeg.itsDirectionH
									  + theFirstLeg.itsDirectionV * theSecondLeg.itsDirectionV;
						if (theDotProduct <= 0.0)
							continue;
						
						//	When the (moving) cue ball strikes the (still) object ball,
						//	energy and momentum will both be conserved if the object
						//	ball carries away the momentum parallel to the line
						//	connecting the balls' centers, while the cue ball carries
						//	away the momentum perpendicular to that line.
						//	Similar triangles give the required final speed
						//	for the cue ball to achieve the desired initial speed
						//	for the object ball.  (Make a sketch and all will be clear!)
						theFinalCueBallSpeed = theSecondLeg.itsStartSpeed / theDotProduct;

						//	Recompute the first leg using the correct final speed.
						SetPath(&theFirstLeg,
								theCueBallH, theCueBallV,
								theFirstLegEndH, theFirstLegEndV,
								theFinalCueBallSpeed);

						//	Is this our best shot so far?
						//	If so, set itsGameOf.Pool2D.itsCueHotSpot.
						if (theFirstLeg.itsStartSpeed < theBestStartSpeed)
						{
							SetCueHotSpotForVector(
								md,
								(md->itsTopology == Topology2DKlein && dv != 0) ?
									- theFirstLeg.itsStartSpeed * theFirstLeg.itsDirectionH / SHOT_STRENGTH :
									+ theFirstLeg.itsStartSpeed * theFirstLeg.itsDirectionH / SHOT_STRENGTH,
								theFirstLeg.itsStartSpeed * theFirstLeg.itsDirectionV / SHOT_STRENGTH);

							theBestStartSpeed = theFirstLeg.itsStartSpeed;
						}
					}
			}
	}
	
	//	Most likely we've found a good shot by now,
	//	but if we haven't, go for Plan B.
	if (theBestStartSpeed == theMaxStartSpeed)
	{
		//	First set up a plausible Plan C,
		//	in case Plan B fails.
		SetCueHotSpotForVector(md, 0.0, theMaxCueVectorLength);
		
		//	Plan B is to find some legal ball and blast it.
		//	Don't worry about colliding with other balls,
		//	but try not to accidentally sink the cue ball.
		for (theBall = 0; theBall < NUM_POOL_BALLS; theBall++)
		{
			if (ObjectBallIsValid(md, theBall))
			{
				//	Set the cue ball's path,
				//	but with a temporary final speed of zero.
				SetPath(&theFirstLeg,
						md->itsGameOf.Pool2D.itsBallPlacement[   0   ].itsH,
						md->itsGameOf.Pool2D.itsBallPlacement[   0   ].itsV,
						md->itsGameOf.Pool2D.itsBallPlacement[theBall].itsH,
						md->itsGameOf.Pool2D.itsBallPlacement[theBall].itsV,
						0.0);
				
				if (PathAvoidsObstacle(&theFirstLeg, 0, 0, HOLE_RADIUS))
				{
					//	Set itsGameOf.Pool2D.itsCueHotSpot in the desired direction,
					//	with maximum speed.
					SetCueHotSpotForVector(
						md,
						theMaxCueVectorLength * theFirstLeg.itsDirectionH,
						theMaxCueVectorLength * theFirstLeg.itsDirectionV);

					//	No need to look at other balls.
					break;
				}
			}
		}
	}

}


static bool ObjectBallIsValid(
	ModelData		*md,
	unsigned int	aBall)
{
	PoolGoal	theGoal;

	//	If aBall isn't in play, for sure it's not valid.
	if ( ! md->itsGameOf.Pool2D.itsBallInPlay[aBall] )
		return false;

	//	Which balls are we shooting for?
	theGoal = (md->itsGameOf.Pool2D.itsWhoseTurn == PoolPlayerA ?
				md->itsGameOf.Pool2D.itsPlayerAGoal : md->itsGameOf.Pool2D.itsPlayerBGoal);

	//	Is the given ball a valid object ball?
	switch (aBall)
	{
		case 0:			//	cue ball
			return false;
		
		case 4:			//	black ball
			if (theGoal == PoolBlack)
				return true;
			else
				return false;
		
		case 1:			//	solid ball
		case 2:
		case 3:
			if (theGoal == PoolUndecided || theGoal == PoolSolids)
				return true;
			else
				return false;
		
		case 5:			//	striped ball
		case 6:
		case 7:
			if (theGoal == PoolUndecided || theGoal == PoolStripes)
				return true;
			else
				return false;
		
		default:	//	should never occur
			return false;
	}
}


static void SetPath(
	BallPath	*aPath,	//	output
	double		h0,		//	initial ball position, in tiling view coordinates,
	double		v0,		//		assumed to lie in 9 central copies of fundamental domain
	double		h1,		//	final ball position, in tiling view coordinates
	double		v1,		//		assumed to lie in 9 central copies of fundamental domain
	double		s1)		//	final ball speed
{
	double	t0;

	aPath->itsStartH		= h0;
	aPath->itsStartV		= v0;

	aPath->itsDirectionH	= h1 - h0;
	aPath->itsDirectionV	= v1 - v0;

	aPath->itsLength		= sqrt(	aPath->itsDirectionH * aPath->itsDirectionH
								  + aPath->itsDirectionV * aPath->itsDirectionV);
	if (aPath->itsLength == 0.0)	//	should never occur
	{
		aPath->itsNormalH	= 0.0;
		aPath->itsNormalV	= 0.0;
		return;
	}

	aPath->itsDirectionH	/= aPath->itsLength;
	aPath->itsDirectionV	/= aPath->itsLength;
	
	aPath->itsNormalH		= - aPath->itsDirectionV;
	aPath->itsNormalV		=   aPath->itsDirectionH;
	
	//	By construction the ball decelerates with a constant BALL_DECELERATION,
	//	so its velocity function is
	//
	//		v(t) = v(0) - BALL_DECELERATION*t
	//
	//	If we take t = 0 to be the time the ball reaches the end
	//	of the path (which admittedly is an odd definition of t = 0,
	//	but makes sense in the present case because that's the time
	//	at which the speed s1 is known), then the velocity function becomes
	//
	//		v(t) = s1 - BALL_DECELERATION*t
	//
	//	Integrating gives the position function
	//
	//		x(t) = x(0) + s1*t - (BALL_DECELERATION/2)*t²
	//
	//	If we measure distances relative to the end of the ball's path,
	//	the position function becomes
	//
	//		x(t) = s1*t - (BALL_DECELERATION/2)*t²
	//
	//	Our plan is to figure out at what (negative) time t the ball
	//	began its trip, and to compute its velocity at that point.
	//
	//		-itsLength = s1*t0 - (BALL_DECELERATION/2)*t0²
	//
	//		(BALL_DECELERATION/2)*t0² - s1*t0 - itsLength = 0
	//
	//		t0 = ( s1 +/- sqrt( s1² + 2*BALL_DECELERATION*itsLength ) )
	//			/ BALL_DECELERATION
	t0 = (s1 - sqrt(s1*s1 + 2*BALL_DECELERATION*aPath->itsLength)) / BALL_DECELERATION;
	aPath->itsStartSpeed = s1 - BALL_DECELERATION*t0;
}


static bool PathIsClear(
	ModelData		*md,
	BallPath		*aPath,			//	assumed to lie within 9 central copies of fundamental domain
	unsigned int	aBall,			//	a ball to ignore when testing for obstructions
	bool			aPocketTest)	//	test the pocket as a potential obstacle? (true on first leg, false on second leg)
{
	//	Would a ball travelling along aPath
	//	avoid other balls and also avoid the pocket?
	
	unsigned int	theBall;
	signed int		dh,
					dv;
	double			theObstacleBallH,
					theObstacleBallV;

	//	Does the path miss all other balls?
	for (theBall = 0; theBall < NUM_POOL_BALLS; theBall++)
	{
		//	Exclude the active ball itself.
		if (theBall == aBall)
			continue;
		
		//	Exclude balls no longer in play.
		if ( ! md->itsGameOf.Pool2D.itsBallInPlay[theBall] )
			continue;
			
		//	Consider the images of theBall within the 9 central copies
		//	of the fundamental domain.
		for (dh = -1; dh <= +1; dh++)
			for (dv = -1; dv <= +1; dv++)
			{
				theObstacleBallH = md->itsGameOf.Pool2D.itsBallPlacement[theBall].itsH + dh;
				theObstacleBallV = md->itsGameOf.Pool2D.itsBallPlacement[theBall].itsV + dv;
				if (md->itsTopology == Topology2DKlein
				 && dv != 0)
				{
					theObstacleBallH = - theObstacleBallH;
				}
				
				if ( ! PathAvoidsObstacle(	aPath,
											theObstacleBallH,
											theObstacleBallV,
											2.0 * BALL_RADIUS) )
					return false;
			}
	}

	//	Does the path miss the pocket?
	if (aPocketTest)
	{
		//	Consider the images of the pocket within the 9 central copies
		//	of the fundamental domain.
		for (dh = -1; dh <= +1; dh++)
			for (dv = -1; dv <= +1; dv++)
				if ( ! PathAvoidsObstacle(aPath, dh, dv, HOLE_RADIUS) )
					return false;
	}

	//	The path is clear.
	return true;
}


static bool PathAvoidsObstacle(
	BallPath	*aPath,
	double		anObstacleH,
	double		anObstacleV,
	double		aRadius)
{
	double	theSidewaysDisplacement,
			theLongwaysDisplacement,
			theFinalH,
			theFinalV,
			dh,
			dv,
			theFinalDistance;

	//	If anObstacle lies too far to either side,
	//	then aPath avoids it.  Compute a dot product to check.
	theSidewaysDisplacement = aPath->itsNormalH * (anObstacleH - aPath->itsStartH)
							+ aPath->itsNormalV * (anObstacleV - aPath->itsStartV);
	if (fabs(theSidewaysDisplacement) >= aRadius)
		return true;

	//	If anObstacle lies behind aPath's start or beyond aPath's end,
	//	then aPath avoids it.  Compute a dot product to check.
	//	Be generous on the far end, keeping in mind that the whole purpose
	//	of this path might be to hit another ball.
	theLongwaysDisplacement = aPath->itsDirectionH * (anObstacleH - aPath->itsStartH)
							+ aPath->itsDirectionV * (anObstacleV - aPath->itsStartV);
	if (theLongwaysDisplacement < 0.0)
	{
		//	On the near end we can be pretty loose, because the ball's initial position
		//	doesn't conflict with any obstacles.
		return true;
	}
	if (theLongwaysDisplacement > aPath->itsLength)
	{
		//	On the far end we need to be a little more careful,
		//	because conflicts at the final position are possible.
		//	The tricky part is that for the "first leg" of a shot,
		//	the final position will, by design, be tangent to another ball.
		theFinalH = aPath->itsStartH  +  aPath->itsLength * aPath->itsDirectionH;
		theFinalV = aPath->itsStartV  +  aPath->itsLength * aPath->itsDirectionV;
		dh = theFinalH - anObstacleH;
		dv = theFinalV - anObstacleV;
		theFinalDistance = sqrt(dh*dh + dv*dv);
		if (theFinalDistance > aRadius - ENDPOINT_EPSILON)
			return true;
	}
	
	//	Report a potential collision.
	return false;
}


static void AddGaussianError(ModelData *md)
{
	//	Add a Gaussian error to itsGameOf.Pool2D.itsCueHotSpot,
	//	which we assume has already been set in preparation
	//	for the computer's shot in human vs. computer mode.
	
	Complex	theExactVector,
			theErrorVector,
			theFinalVector;
	double	theLength;

	static double	theStandardDeviation[4] = {0.04, 0.03, 0.02, 0.00};
	
	//	Think of itsGameOf.Pool2D.itsCueHotSpot.itsHV as a complex number
	//	and multiply by a Gaussian random complex number
	//	with mean 1 and standard deviation depending
	//	on the difficulty level.

	CueVector(md, &theExactVector.re, &theExactVector.im, &theLength);
	
	GEOMETRY_GAMES_ASSERT(md->itsDifficultyLevel < 4, "invalid difficulty level");

	theErrorVector.re = 1.0 + theStandardDeviation[md->itsDifficultyLevel] * GetRandomGaussian();
	theErrorVector.im =       theStandardDeviation[md->itsDifficultyLevel] * GetRandomGaussian();

	theFinalVector = ComplexMultiply(theExactVector, theErrorVector);
	
	//	Don't exceed length 0.5 - CUE_DEAD_ZONE.
	theLength = sqrt(theFinalVector.re * theFinalVector.re
				   + theFinalVector.im * theFinalVector.im);
	if (theLength > 0.5 - CUE_DEAD_ZONE)
	{
		theFinalVector.re *= (0.5 - CUE_DEAD_ZONE) / theLength;
		theFinalVector.im *= (0.5 - CUE_DEAD_ZONE) / theLength;
	}

	//	Set itsGameOf.Pool2D.itsCueHotSpot to the final value.
	SetCueHotSpotForVector(md, theFinalVector.re, theFinalVector.im);
}


static double GetRandomGaussian(void)
{
	//	Return a Gaussian random number with mean 0 and standard deviation 1.

	//	Thanks to Robin Lock for telling me how to do this.

	double	u1,	//	uniform on (0,1]
			u2,	//	uniform on [0,1]
			x;	//	Gaussian with mean 0 and standard deviation 1

	do
	{
		u1 = RandomFloat();
	} while (u1 == 0.0);

	u2 = RandomFloat();

	x = sqrt(-2*log(u1)) * cos(2*PI*u2);

	return x;
}


static Complex ComplexMultiply(
	Complex	z,
	Complex	w)
{
	Complex zw;
	
	zw.re = z.re * w.re
		  - z.im * w.im;

	zw.im = z.re * w.im
		  + z.im * w.re;
	
	return zw;
}


static void PoolSimulationUpdate(ModelData *md)
{
	switch (md->itsSimulationStatus)
	{
		case Simulation2DPoolRollBalls:
		
			//	Advance the simulation by one frame.
			Shoot2(md, md->itsSimulationDeltaTime);

			//	Have the balls come to rest?
			if ( ! BallsAreRolling(md) )
			{
				SimulationEnd(md);
				Shoot3(md);
			}
			break;
		
		case Simulation2DPoolPlaceCueStick:

			//	PoolDrawGL() will check md->itsSimulationStatus
			//	and md->itsSimulationElapsedTime and scale
			//	the cue stick length accordingly.

			//	When the cue stick animation is finished,
			//	pause so that the user may appreciate it.
			if (md->itsSimulationElapsedTime >= STICK_PLACE_TIME)
			{
				SimulationEnd(md);
				SimulationBegin(md, Simulation2DPoolPauseBeforeShooting);
			}

			break;
		
		case Simulation2DPoolPauseBeforeShooting:

			//	After displaying the ready-to-go pool cue for one second...
			if (md->itsSimulationElapsedTime >= STICK_PAUSE_TIME)
			{
				SimulationEnd(md);	//	...terminate the waiting period and
				Shoot1(md);			//	...let the balls roll!
			}

			break;

		default:
			break;
	}
}


unsigned int GetNum2DPoolBackgroundTextureRepetitions(void)
{
	return NUM_2D_BACKGROUND_TEXTURE_REPETITIONS_POOL;
}

void Get2DPoolKleinAxisColors(
	float	someKleinAxisColors[2][4])	//	output;  premultiplied alpha
{
	someKleinAxisColors[0][0] = 1.00;
	someKleinAxisColors[0][1] = 1.00;
	someKleinAxisColors[0][2] = 0.00;
	someKleinAxisColors[0][3] = 1.00;

	someKleinAxisColors[1][0] = 0.50;
	someKleinAxisColors[1][1] = 1.00;
	someKleinAxisColors[1][2] = 1.00;
	someKleinAxisColors[1][3] = 1.00;
}


unsigned int GetNum2DPoolSprites(void)
{
	return NUM_POOL_BALLS + 4;	//	8 balls, pocket, stick, line-of-sight and hot spot
}

void Get2DPoolSpritePlacements(
	ModelData		*md,				//	input
	unsigned int	aNumSprites,		//	input
	Placement2D		*aPlacementBuffer)	//	output;  big enough to hold aNumSprites Placement2D's
{
	unsigned int	i;
	double			theCueVectorH,
					theCueVectorV,
					theCueVectorLength,
					theDirectionH,
					theDirectionV,
					theNearEnd,
					theFarEnd;

	GEOMETRY_GAMES_ASSERT(
		md->itsGame == Game2DPool,
		"Game2DPool must be active");

	GEOMETRY_GAMES_ASSERT(
		aNumSprites == GetNum2DPoolSprites(),
		"Internal error:  wrong buffer size");

	//	balls
	for (i = 0; i < NUM_POOL_BALLS; i++)
	{
		if (md->itsGameOf.Pool2D.itsBallInPlay[i])
			aPlacementBuffer[i] = md->itsGameOf.Pool2D.itsBallPlacement[i];
		else
			aPlacementBuffer[i] = gUnusedPlacement;
	}
	
	//	pocket
	aPlacementBuffer[NUM_POOL_BALLS + 0].itsH		= 0.0;
	aPlacementBuffer[NUM_POOL_BALLS + 0].itsV		= 0.0;
	aPlacementBuffer[NUM_POOL_BALLS + 0].itsFlip	= false;
	aPlacementBuffer[NUM_POOL_BALLS + 0].itsAngle	= 0.0;
	aPlacementBuffer[NUM_POOL_BALLS + 0].itsSizeH	= 2.0 * HOLE_RADIUS;
	aPlacementBuffer[NUM_POOL_BALLS + 0].itsSizeV	= 2.0 * HOLE_RADIUS;
	
	//	cue stick
	if ( ! md->itsGameIsOver
	 && md->itsGameOf.Pool2D.itsCueStatus != PoolCueNone)
	{
		CueVector(md, &theCueVectorH, &theCueVectorV, &theCueVectorLength);

		if (theCueVectorLength > 0.0)
		{
			theDirectionH = theCueVectorH / theCueVectorLength;
			theDirectionV = theCueVectorV / theCueVectorLength;
			
			if (md->itsSimulationStatus == Simulation2DPoolPlaceCueStick
			 && md->itsSimulationElapsedTime <= STICK_PLACE_TIME)
			{
				theCueVectorLength *= md->itsSimulationElapsedTime / STICK_PLACE_TIME;
			}

			theNearEnd	= CUE_DEAD_ZONE;
			theFarEnd	= CUE_DEAD_ZONE  +  STICK_FACTOR * theCueVectorLength;

			//	stick

			aPlacementBuffer[NUM_POOL_BALLS + 1].itsH		= md->itsGameOf.Pool2D.itsBallPlacement[0].itsH  -  ((theNearEnd + theFarEnd)/2) * theDirectionH;
			aPlacementBuffer[NUM_POOL_BALLS + 1].itsV		= md->itsGameOf.Pool2D.itsBallPlacement[0].itsV  -  ((theNearEnd + theFarEnd)/2) * theDirectionV;
			aPlacementBuffer[NUM_POOL_BALLS + 1].itsFlip	= false;
			aPlacementBuffer[NUM_POOL_BALLS + 1].itsAngle	= atan2(theDirectionV, theDirectionH);
			aPlacementBuffer[NUM_POOL_BALLS + 1].itsSizeH	= theFarEnd - theNearEnd;
			aPlacementBuffer[NUM_POOL_BALLS + 1].itsSizeV	= 0.02;

			Normalize2DPlacement(&aPlacementBuffer[NUM_POOL_BALLS + 1], md->itsTopology);
			
			//	line of sight

			aPlacementBuffer[NUM_POOL_BALLS + 2].itsH		= md->itsGameOf.Pool2D.itsBallPlacement[0].itsH  +  ((theNearEnd + theFarEnd)/2) * theDirectionH;
			aPlacementBuffer[NUM_POOL_BALLS + 2].itsV		= md->itsGameOf.Pool2D.itsBallPlacement[0].itsV  +  ((theNearEnd + theFarEnd)/2) * theDirectionV;
			aPlacementBuffer[NUM_POOL_BALLS + 2].itsFlip	= false;
			aPlacementBuffer[NUM_POOL_BALLS + 2].itsAngle	= atan2(theDirectionV, theDirectionH);
			aPlacementBuffer[NUM_POOL_BALLS + 2].itsSizeH	= theFarEnd - theNearEnd;
			aPlacementBuffer[NUM_POOL_BALLS + 2].itsSizeV	= 0.01;

			Normalize2DPlacement(&aPlacementBuffer[NUM_POOL_BALLS + 2], md->itsTopology);

#if defined(TORUS_GAMES_2D_TOUCH_INTERFACE) && ! defined(GAME_CONTENT_FOR_SCREENSHOT)
			//	Draw the cue stick's hot spot, if appropriate.
			if (md->itsGameOf.Pool2D.itsCueStatus == PoolCueActive)
			{
				aPlacementBuffer[NUM_POOL_BALLS + 3].itsH		= md->itsGameOf.Pool2D.itsCueHotSpot.itsH;
				aPlacementBuffer[NUM_POOL_BALLS + 3].itsV		= md->itsGameOf.Pool2D.itsCueHotSpot.itsV;
				aPlacementBuffer[NUM_POOL_BALLS + 3].itsFlip	= false;
				aPlacementBuffer[NUM_POOL_BALLS + 3].itsAngle	= 0.0;
				aPlacementBuffer[NUM_POOL_BALLS + 3].itsSizeH	= CUE_HOT_SPOT_SIZE;
				aPlacementBuffer[NUM_POOL_BALLS + 3].itsSizeV	= CUE_HOT_SPOT_SIZE;
		
				Normalize2DPlacement(&aPlacementBuffer[NUM_POOL_BALLS + 3], md->itsTopology);
			}
			else
#endif
			{
				aPlacementBuffer[NUM_POOL_BALLS + 3] = gUnusedPlacement;
			}
		}
		else
		{
			//	The cue stick has zero length.
			//	Really it doesn't need to be drawn at all,
			//	but to avoid messy communication between this function
			//	and the platform-specific drawing code,
			//	let's just set a zero-size placement.
			//	The platform-specific drawing code will draw
			//	the stick and line of sight, but the user won't see them.
			aPlacementBuffer[NUM_POOL_BALLS + 1] = gZeroPlacement;
			aPlacementBuffer[NUM_POOL_BALLS + 2] = gZeroPlacement;
			aPlacementBuffer[NUM_POOL_BALLS + 3] = gZeroPlacement;
		}
	}
	else
	{
		aPlacementBuffer[NUM_POOL_BALLS + 1] = gUnusedPlacement;
		aPlacementBuffer[NUM_POOL_BALLS + 2] = gUnusedPlacement;
		aPlacementBuffer[NUM_POOL_BALLS + 3] = gUnusedPlacement;
	}
}
